/**
 * \file: ald.c
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * \component: authorization level daemon
 *
 * \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
 *
 * \copyright (c) 2017 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/
#include <stdbool.h>
#include <stdlib.h>
#include <string.h>
#include <gio/gio.h>
#include <glib-unix.h>
#include <errno.h>
#include <signal.h>

#include "util/logger.h"
#include "app_iface/app_iface.h"
#include "util/version_info.h"
#include "control/configuration.h"
#include "control/daemon_fsm.h"
#include "control/levelchanger.h"
#include "model/level_configuration.h"
#include "model/daemon_model.h"
#include "model/signature_db.h"
#include "model/challenge.h"
#include "ald_types.h"

typedef struct ald_handle_timeout_t
{
	gint challenge_source_id;
	gint script_exec_source_id;
}ald_handle_timeout_t;

static GMainLoop *g_main_loop_ctx=NULL;
static GSource *levelchanger_event_src=NULL;
static error_code_t daemon_exit_result=RESULT_OK;
static ald_handle_timeout_t ald_handle_timeout={0};

static void ald_init_logging(int argc, char *argv[]);
static error_code_t ald_init_configuration(int argc, char *argv[]);
static error_code_t ald_setup_mainloop_and_app_iface(void);
static error_code_t ald_init_levelchanger(void);

static void ald_early_parse_args(int argc,char *argv[],
		const char **proc_name, logger_loglevel_t *loglevel, bool *console_enabled);

static void ald_sync_app_iface_with_model(void);

static error_code_t ald_register_sighandler(void);
static gboolean ald_sig_handler (gpointer data);

static gboolean ald_on_levelchanger_event(gpointer user_data);

static gboolean ald_on_challenge_timeout(gpointer user_data);

static gboolean ald_on_script_exec_timeout(gpointer user_data);

//use our own implementation if we are faced with a to old version of glib
#if !GLIB_CHECK_VERSION (2, 36, 0)
static GSourceFuncs source_funcs={0};
static GPollFD source_pollfd={0};

static GSource *g_unix_fd_source_new(gint fd, GIOCondition condition);
static gboolean ald_unix_fd_source_prepare(GSource *source, gint *timeout);
static gboolean ald_unix_fd_source_check (GSource *source);
static gboolean ald_unix_fd_source_dispatch(GSource* source, GSourceFunc callback, gpointer user_data);

#endif

//---------------------------------------- API members ---------------------------------------------
error_code_t ald_init(int argc, char *argv[])
{
	error_code_t result;

#if !GLIB_CHECK_VERSION (2, 35, 0)
	g_type_init ();
#endif

	//start logging
	ald_init_logging(argc,argv);

	//register signal handlers (no dependencies)
	result=ald_register_sighandler();

	//parse the configuration (logging needs to be up)
	if (result==RESULT_OK)
		result=ald_init_configuration(argc,argv);

	//setting up the model (needs logging)
	if (result==RESULT_OK)
		result=daemon_model_init();

	//loading signatures
	if (result==RESULT_OK)
		result=signature_db_load();

	//creating a list of available level configurations
	if (result==RESULT_OK)
		result=level_configuration_init(configuration_get_script_root_dir());

	//setting up the main loop and the application interface (logging required)
	if (result==RESULT_OK)
		result=ald_setup_mainloop_and_app_iface();

	//start the level changer (depends on a mainloop and the logger)
	if (result==RESULT_OK)
		result=ald_init_levelchanger();

	//updating the app_iface with the model (needs a mainloop and the model up and running)
	if (result==RESULT_OK)
		ald_sync_app_iface_with_model();

	return result;
}

error_code_t ald_mainloop(void)
{
	logger_log_debug("ALD - Entering the main loop now.");

	// daemon_exit_result is set again in case of a shutdown request caused by errors.
	daemon_exit_result=RESULT_OK;

	//we are close to entering the mainloop. Lets kick of the daemon_fsm, which might directly result
	//in a recovery kicked off
	daemon_fsm_kickoff();

	g_main_loop_run(g_main_loop_ctx);

	logger_log_debug("ALD - Main loop left. Shutting down now.");
	return daemon_exit_result;
}

void ald_shutdown(void)
{
	if (g_main_loop_ctx!=NULL)
		g_main_loop_quit(g_main_loop_ctx);
}

void ald_deinit(void)
{
	logger_log_info("Authorization Level Daemon exiting");

	//it is important that the deinit member of the levelchanger is called first. This ensures that a possibly running
	//thread keeps of its hand from configuration data and the signature db while they are removed from memory.
	levelchanger_deinit();
	if (levelchanger_event_src!=NULL)
	{
		g_source_destroy(levelchanger_event_src);
		g_source_unref(levelchanger_event_src);
		levelchanger_event_src=NULL;
	}

	app_iface_deinit();

	signature_db_deinit();

	level_configuration_deinit();

	daemon_model_deinit();

	if (g_main_loop_ctx!=NULL)
	{
		g_main_loop_unref(g_main_loop_ctx);
		g_main_loop_ctx=NULL;
	}

	logger_deinit();
	configuration_deinit();
}

void ald_die_on_resource_issues(void)
{
	logger_log_error("Authorization Level Daemon is faced some resources problems. Probably we ran "
			"out of memory. Best solution is now to die bravely.");
	exit(ENOMEM);
}

void ald_request_daemon_exit_due_to_errors(error_code_t error)
{
	/* don't overwrite the error code of the first error */
	if (daemon_exit_result == RESULT_OK) {
		daemon_exit_result=error;
		logger_log_debug("The daemon has been requested to shutdown due to runtime errors. Code: %d",(int)error);
	}

	/* in case of multiple errors we send the shutdown trigger again */
	daemon_fsm_shutdown_request();
}

void ald_activate_script_exec_timeout(void)
{
	if (g_main_loop_ctx!=NULL)
	{
		ald_handle_timeout.script_exec_source_id = g_timeout_add(configuration_get_script_exec_timeout_ms(),ald_on_script_exec_timeout,NULL);
		logger_log_debug("ALD - Activating a timeout for the script execution %d ms.",
				configuration_get_script_exec_timeout_ms());
	}
}

void ald_deactivate_script_exec_timeout(void)
{
	if(ald_handle_timeout.script_exec_source_id > 0)
	{
		(void)g_source_remove (ald_handle_timeout.script_exec_source_id);
		ald_handle_timeout.script_exec_source_id = 0;
		logger_log_debug("ALD - Deactivating the timeout for the script execution.");
	}
}

void ald_activate_challenge_timeout(void)
{
	if (g_main_loop_ctx!=NULL)
	{
		ald_handle_timeout.challenge_source_id = g_timeout_add(configuration_get_challenge_timeout_ms(),ald_on_challenge_timeout,NULL);
		logger_log_debug("ALD - Activating a timeout for the current challenge of %d ms.",
				configuration_get_challenge_timeout_ms());
	}
}

void ald_deactivate_challenge_timeout(void)
{
	/*Fix for SWGIII-10742 : Remove the corresponding timeout when the challenge is invalidated.*/
	if(ald_handle_timeout.challenge_source_id > 0)
	{
		(void)g_source_remove (ald_handle_timeout.challenge_source_id);
		ald_handle_timeout.challenge_source_id = 0;
	}
	logger_log_debug("ALD - Deactivating the timeout for the current challenge.");
}
//-------------------------------------------------------------------------------------------------

//------------------------------------- private members -------------------------------------------
static void ald_sync_app_iface_with_model(void)
{
	app_iface_signal_level_changed(daemon_model_get_active_level());
}

static gboolean ald_sig_handler (gpointer data)
{
	(void)data;

	logger_log_debug("Cought SIGTERM, SIGHUP, or SIGINT.");
	daemon_fsm_signal_shutdown_request();

	return  G_SOURCE_CONTINUE;
}

static error_code_t ald_register_sighandler(void)
{
	error_code_t result=RESULT_OK;

	logger_log_debug("ALD - Registering signal handlers.");

	(void)g_unix_signal_add(SIGTERM, ald_sig_handler, NULL);
	(void)g_unix_signal_add(SIGINT, ald_sig_handler, NULL);
	(void)g_unix_signal_add(SIGHUP, ald_sig_handler, NULL);

	return result;
}

static void ald_early_parse_args(int argc,char *argv[],
		const char **proc_name, logger_loglevel_t *loglevel, bool *console_enabled)
{
	int i;
	*proc_name=argv[0];

	for (i=1;i<argc;i++)
	{
		if (strcmp(argv[i],"-q")==0 || strcmp(argv[i],"--quiet")==0)
			*console_enabled=false;
		if (strcmp(argv[i],"-l")==0 || strcmp(argv[i],"--loglevel")==0)
		{
			if (i<argc-1)
				logger_parse_loglevel(argv[i+1],loglevel);
		}
		if (strstr(argv[i],"--loglevel=")!=NULL)
		{
			const char *value;
			value=strchr(argv[i],'=');
			//to satisfy lin(t),sorry
			if (value!=NULL)
			{
				if (value[1]!='\0')
					logger_parse_loglevel(value,loglevel);
			}
		}
	}
}

static void ald_init_logging(int argc, char *argv[])
{
	const char *proc_name;
	logger_loglevel_t loglevel;
	bool console_enabled;

	//get default values
	loglevel=configuration_get_loglevel();
	console_enabled=configuration_is_console_enabled();

	//override them in case command line parameters are passed
	ald_early_parse_args(argc,argv, &proc_name, &loglevel, &console_enabled);
	logger_init(proc_name,loglevel,console_enabled);

	//say hello
	logger_log_status("%s",VERSION_INFO_FORMATED_LINE);
}

static error_code_t ald_init_configuration(int argc, char *argv[])
{
	error_code_t result;
	result=configuration_init(argc,argv);

	//reads out the configuration file (either the default one or the one given as command line option)
	if (result==RESULT_OK)
		result=configuration_parse_ald_configuration_file();

	//set loglevel again, command line options or the configuration file might have changed the initial value
	logger_set_log_level(configuration_get_loglevel());
	logger_set_console_enabled(configuration_is_console_enabled());

	return result;
}

static error_code_t ald_setup_mainloop_and_app_iface(void)
{
	g_main_loop_ctx = g_main_loop_new(NULL, FALSE);
	return app_iface_init();
}

static error_code_t ald_init_levelchanger(void)
{
	error_code_t result;

	challenge_seed_random_number_generator();

	result=levelchanger_init();
	if (result==RESULT_OK)
	{
		levelchanger_event_src=g_unix_fd_source_new(levelchanger_get_pollfd(),G_IO_IN);
		g_source_set_callback(levelchanger_event_src,ald_on_levelchanger_event,NULL,NULL);
		(void)g_source_attach(levelchanger_event_src,NULL);
	}
	return result;
}

static gboolean ald_on_levelchanger_event(gpointer user_data)
{
	(void)user_data;
	levelchanger_on_event();
	return TRUE;
}

static gboolean ald_on_script_exec_timeout(gpointer user_data)
{
	(void)user_data;
	int state = 0;
	pid_t childprc;
	pid_t ret;

	childprc  = levelchanger_get_script_exec_prc_id();
	ret = waitpid(childprc, &state, WNOHANG);
	if(ret == 0)
	{
		levelchanger_set_result_to_script_exec_timeout();
		logger_log_error("Script execution timeout expired. Stopping it.");
		kill(childprc, SIGKILL);
	}
	ald_handle_timeout.script_exec_source_id = 0;

	//we need the timeout timer only once
	return FALSE;
}

static gboolean ald_on_challenge_timeout(gpointer user_data)
{
	(void)user_data;
	if (challenge_is_valid())
		logger_log_info("Challenge timeout expired. Invalidating current challenge.");
	/*Fix for SWGIII-10742: Need not remove the timeout as the timeout is destroyed when this function returns FALSE*/
	ald_handle_timeout.challenge_source_id = 0;
	challenge_invalidate();

	//we need the timeout timer only once
	return FALSE;
}
//-------------------------------------------------------------------------------------------------


//------------------------------------- implementation of g_unix_fd_source (for glib < 2.36) -------
//use our own implementation if we are faced with a to old version of glib
#if !GLIB_CHECK_VERSION (2, 36, 0)

static GSource * g_unix_fd_source_new(gint fd, GIOCondition condition)
{
	GSource *new_source;

	source_pollfd.fd = fd;
	source_pollfd.events = condition;

	source_funcs.prepare=ald_unix_fd_source_prepare;
	source_funcs.check=ald_unix_fd_source_check;
	source_funcs.dispatch=ald_unix_fd_source_dispatch;

	new_source=g_source_new(&source_funcs, sizeof(GSource));
	g_source_add_poll(new_source, &source_pollfd);

	return new_source;
}

static gboolean ald_unix_fd_source_prepare(GSource *source, gint *timeout)
{
	(void)source;
	*timeout = -1;
	return FALSE;
}

static gboolean ald_unix_fd_source_check (GSource *source)
{
	(void)source;
	return source_pollfd.revents!=0;
}

static gboolean ald_unix_fd_source_dispatch(GSource* source, GSourceFunc callback, gpointer user_data)
{
	(void)source;
	if (callback != NULL)
		return callback(user_data);
	return TRUE;
}

#endif
